/*********************************************************************/
/*                                                                   */
/*  This Program Written by Paul Edwards, 3:711/934@fidonet.         */
/*  Released to the Public Domain                                    */
/*                                                                   */
/*********************************************************************/
/*********************************************************************/
/*                                                                   */
/*  rsend - front-end mailer                                         */
/*                                                                   */
/*  This program will poll your "boss" for mail and then end.        */
/*                                                                   */
/*********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <pdcomm.h>
#include <crcxm.h>
#include <m2x.h>
#include <error.h>
#include <zmcore.h>
#include <zmext.h>
#include <faddr.h>
#include <unused.h>
#include <strinl.h>
#include <stritr.h>

#if defined(AMIGA)
#include "sleepa.h"
#elif !defined(MSDOS)
#include <os2.h>
#else
 /* #include <dos.h> */
#endif

#define RSEND_USAGE "RSN001 Usage: rsend <config file>\n"
#define RSEND_FAIL_CONNECT "RSN002 Failed to connect to remote\n"

typedef struct {
    char port[200];
    long baud;
    PDCOMM pdcomm;
    ZMCORE zmcore;
    ZMEXT zmext;
    fidoAddress address;
    fidoAddress boss;
    char bossphone[50];
    char bosspwd[50];
    char system[50];
    char sysop[50];
    char outbound[FILENAME_MAX];
    char *config_file;
} RSEND;

static void delaySeconds(int secs);

void rsendDefaults(RSEND *rsend);
void rsendInit(RSEND *rsend);
void rsendRun(RSEND *rsend, int argc, char **argv);
void rsendTerm(RSEND *rsend);

static void rsendReadConfig(RSEND *rsend);
static void rsendPoll(RSEND *rsend);
static void rsendPoll2(RSEND *rsend);
static void rsendDial(RSEND *rsend);
static void rsendYoohoo(RSEND *rsend);
static void rsendSendHello(RSEND *rsend);
static void rsendSendZmodem(RSEND *rsend);
static void rsendReceiveZmodem(RSEND *rsend);
static void delaySeconds(int secs);

int main(int argc, char **argv)
{
    static RSEND rsend;
    
    errorDefaults();
    errorInit();
    if (ALLOK)
    {
        rsendDefaults(&rsend);
        rsendInit(&rsend);
        if (ALLOK)
        {
            rsendRun(&rsend, argc, argv);
            rsendTerm(&rsend);
        }
    }
    errorTerm();
    return (0);
}

void rsendDefaults(RSEND *rsend)
{
    unused(rsend);
    return;
}

void rsendInit(RSEND *rsend)
{
    rsend->baud = 0;
    return;
}

void rsendRun(RSEND *rsend, int argc, char **argv)
{
    if (argc < 2)
    {
        errorSet(RSEND_USAGE);
    }
    else
    {
        rsend->config_file = *(argv+1);
        rsendReadConfig(rsend);
        if (ALLOK)
        {
            rsendPoll(rsend);
        }
    }
    return;
}

static void rsendReadConfig(RSEND *rsend)
{
    FILE *fp;
    char buf[200];
    static char *keywords[] = { 
        "Port",
        "Baud",
        "System",
        "Sysop",
        "Address",
        "Boss",
        "BossPhone",
        "BossPwd",
        "Outbound",
        NULL
    };
    int x;
    size_t len;
    char *keyword;
    FADDR faddr;
    char *start;
    
    fp = fopen(rsend->config_file, "r");
    while (fgets(buf, sizeof buf, fp) != NULL)
    {
        strinl(buf);
        stritr(buf);
        for (x = 0; keywords[x] != NULL; x++)
        {
            len = strlen(keywords[x]);
            start = buf + len + 1;
            keyword = keywords[x];
            if ((strncmp(buf, keyword, len) == 0)
                && isspace(buf[len]))
            {
                if (strcmp(keyword, "Port") == 0)
                {
                    strcpy(rsend->port, start);
                }
                else if (strcmp(keyword, "Baud") == 0)
                {
                    rsend->baud = atol(start);
                }
                else if (strcmp(keyword, "System") == 0)
                {
                    strcpy(rsend->system, start);
                }
                else if (strcmp(keyword, "Sysop") == 0)
                {
                    strcpy(rsend->sysop, start);
                }
                else if (strcmp(keyword, "Address") == 0)
                {
                    faddrOnePassCreate(&faddr, start, &rsend->address);
                }
                else if (strcmp(keyword, "Boss") == 0)
                {
                    faddrOnePassCreate(&faddr, start, &rsend->boss);
                }
                else if (strcmp(keyword, "BossPhone") == 0)
                {
                    strcpy(rsend->bossphone, start);
                }
                else if (strcmp(keyword, "BossPwd") == 0)
                {
                    strcpy(rsend->bosspwd, start);
                }
                else if (strcmp(keyword, "Outbound") == 0)
                {
                    strcpy(rsend->outbound, start);
                }
            }
        }
    }
    fclose(fp);
    return;
}

static void rsendPoll(RSEND *rsend)
{
    pdcommDefaults(&rsend->pdcomm);
    pdcommInit(&rsend->pdcomm, rsend->port);
    if (rsend->baud != 0)
    {
        pdcommSetSpeed(&rsend->pdcomm, rsend->baud);
    }
    if (ALLOK)
    {
        rsendPoll2(rsend);
    }
    pdcommTerm(&rsend->pdcomm);
    return;
}

void rsendTerm(RSEND *rsend)
{
    unused(rsend);
    return;
}

static void rsendPoll2(RSEND *rsend)
{
    zmextDefaults(&rsend->zmext);
    zmcoreDefaults(&rsend->zmcore);
    zmextInit(&rsend->zmext, &rsend->pdcomm);
    zmcoreInit(&rsend->zmcore, &rsend->zmext);
    rsendDial(rsend);
    rsendYoohoo(rsend);
    rsendSendZmodem(rsend);
    if (ALLOK)
    {
        rsendReceiveZmodem(rsend);
    }
    zmcoreTerm(&rsend->zmcore);
    zmextTerm(&rsend->zmext);
    return;
}

static void rsendDial(RSEND *rsend)
{
    char buf[50];
    int ch;
    int live = 0;
#ifndef LOCAL
    int x;
#endif

    delaySeconds(1);
    sprintf(buf, "ATDT%s\r", rsend->bossphone);
    printf("sending string %s\n", buf);
    pdcommWriteBuf(&rsend->pdcomm, (unsigned char *)buf, strlen(buf));
#ifndef LOCAL
    printf("waiting for connect\n");
    delaySeconds(1);
    /* clear inbound */
    for (x = 0; x < 500; x++)
    {
        if (pdcommReadCh(&rsend->pdcomm) == EOF) break;
    }
    for (x=0; x < 50;)
    {
        if ((ch = pdcommReadCh(&rsend->pdcomm)) == 'C')
        {
            live = 1;
            printf("%c", ch);
        }
        else if ((ch == 'O') && live)
        {
            printf("%c", ch);
            printf("got connect\n");
            break;
        }
        else if (ch > 0)
        {
            printf("%c", ch);
            live = 0;
        }
        else
        {
            delaySeconds(1);
            live = 0;
            x++;
        }
    }
    if (x == 30)
    {
        printf("failed to connect\n");
        errorSet(RSEND_FAIL_CONNECT);
    }
#endif    
    return;
}

#define SM_SEND_YOOHOO 0x00
#define SM_QUIT_YOOHOO 0x01
#define SM_WAIT_ENQ 0x02
#define SM_SEND_HELLO 0x03
#define SM_WAIT_HELLO_ACK 0x04
#define SM_WAIT_HIS_HELLO 0x05
#define SM_SEND_ENQ 0x06
#define SM_WAIT_START 0x07
#define SM_ACK_HIS_HELLO 0x08
#define SM_WAIT_HIS_YOOHOO 0x09

static void rsendYoohoo(RSEND *rsend)
{
    int ch;
    int state = SM_SEND_YOOHOO;
    int numchars = 0;
    int numsecs = 0;

    /* wait for 60 seconds or 2000 characters */
    
    while ((state != SM_QUIT_YOOHOO) && (numsecs < 60))
    {
        switch (state)
        {
            case SM_SEND_YOOHOO:
                printf("in send yoohoo\n");
                pdcommWriteCh(&rsend->pdcomm, 0xf1);
                state = SM_WAIT_ENQ;
                break;

            case SM_WAIT_ENQ:
                printf("in wait enq\n");
                while ((ch = pdcommReadCh(&rsend->pdcomm)) > 0)
                {
                    numchars++;
                    if (ch == 0x05)
                    {
                        state = SM_SEND_HELLO;
                        break;
                    }
                    else if (numchars > 2000)
                    {
                        state = SM_QUIT_YOOHOO;
                        break;
                    }
                }
                if (state == SM_WAIT_ENQ)
                {
                    delaySeconds(1);
                    numsecs++;
                    state = SM_SEND_YOOHOO;
                }
                break;
                
            case SM_SEND_HELLO:
                printf("in send hello\n");
                rsendSendHello(rsend);
                state = SM_WAIT_HELLO_ACK;
                break;

            case SM_WAIT_HELLO_ACK:
                printf("in wait hello ack\n");
                while ((ch = pdcommReadCh(&rsend->pdcomm)) > 0)
                {
                    numchars++;
                    if (ch == 0x06)
                    {
                        state = SM_WAIT_HIS_YOOHOO;
                        break;
                    }
                    else if (numchars > 2000)
                    {
                        state = SM_QUIT_YOOHOO;
                        break;
                    }
                }
                if (state == SM_WAIT_HELLO_ACK)
                {
                    delaySeconds(1);
                    numsecs++;
                    state = SM_SEND_HELLO;
                }
                break;
                
            case SM_WAIT_HIS_YOOHOO:
                printf("in wait his yoohoo\n");
                while ((ch = pdcommReadCh(&rsend->pdcomm)) > 0)
                {
                    numchars++;
                    if (ch == 0xf1)
                    {
                        state = SM_SEND_ENQ;
                        break;
                    }
                    else if (numchars > 2000)
                    {
                        state = SM_QUIT_YOOHOO;
                        break;
                    }
                }
                if (state == SM_WAIT_HIS_YOOHOO)
                {
                    delaySeconds(1);
                    numsecs++;
                }
                break;
                
            case SM_SEND_ENQ:
                printf("in send enq\n");
                pdcommWriteCh(&rsend->pdcomm, 0x05);
                state = SM_WAIT_START;
                break;
                
            case SM_WAIT_START:
                printf("in wait start\n");
                delaySeconds(1);
                numsecs++;
                while ((ch = pdcommReadCh(&rsend->pdcomm)) > 0)
                {
                    numchars++;
                    if (ch == 0x1f)
                    {
                        state = SM_ACK_HIS_HELLO;
                        break;
                    }
                    else if (ch == 0xf1)
                    {
                        /* must have missed my ENQ */
                        state = SM_SEND_ENQ;
                        break;
                    }
                    else if (numchars > 2000)
                    {
                        state = SM_QUIT_YOOHOO;
                        break;
                    }
                }
                if (state == SM_WAIT_START)
                {
                    delaySeconds(1);
                    numsecs++;
                    state = SM_SEND_ENQ;
                }
                break;
                
            case SM_ACK_HIS_HELLO:
                printf("in ack his hello\n");
                pdcommWriteCh(&rsend->pdcomm, 0x06);
                pdcommWriteCh(&rsend->pdcomm, 0x06);
                pdcommWriteCh(&rsend->pdcomm, 0x06);
                state = SM_QUIT_YOOHOO;
                break;
        }
    }
    delaySeconds(1);
    return;
}
                
static void rsendSendHello(RSEND *rsend)
{
    static unsigned char buf[128];
    CRCXM crcxm;
    int x;

    pdcommWriteCh(&rsend->pdcomm, 0x1f);
    memset(buf, '\0', sizeof buf);
    buf[0] = 0x6f;
    buf[2] = 0x01;
    strcpy((char *)(buf + 10), rsend->system);
    strcpy((char *)(buf + 70), rsend->sysop);
    buf[90] = rsend->address.zone & 0xff;
    buf[91] = (rsend->address.zone >> 8) & 0xff;
    buf[92] = rsend->address.net & 0xff;
    buf[93] = (rsend->address.net >> 8) & 0xff;
    buf[94] = rsend->address.node & 0xff;
    buf[95] = (rsend->address.node >> 8) & 0xff;
    buf[96] = rsend->address.point & 0xff;
    buf[97] = (rsend->address.point >> 8) & 0xff;

    strcpy((char *)(buf + 98), rsend->bosspwd);

    buf[114] = 0x04;
    pdcommWriteBuf(&rsend->pdcomm, buf, sizeof buf);
    crcxmInit(&crcxm);
    for (x = 0; x < 128; x++)
    {
        crcxmUpdate(&crcxm, buf[x]);
    }
    pdcommWriteCh(&rsend->pdcomm, crcxmHighbyte(&crcxm));
    pdcommWriteCh(&rsend->pdcomm, crcxmLowbyte(&crcxm));

    return;
}

static void rsendSendZmodem(RSEND *rsend)
{
    zmextFileSetSpec(&rsend->zmext, rsend->outbound);
    zmcoreSend(&rsend->zmcore);
    return;
}

static void rsendReceiveZmodem(RSEND *rsend)
{
    zmcoreReceive(&rsend->zmcore);
    return;
}

static void delaySeconds(int secs)
{
#if defined(AMIGA)
    AmiSleep(secs);
#elif !defined(MSDOS)
    DosSleep(secs * 1000L);
#else
    sleep(secs);
#endif
    return;
}    

